/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: BehaviourComposerLearnerStarterAction.java,v 1.21 2006/07/25 04:53:06 steven Exp $$ */
package org.lamsfoundation.lams.tool.behaviourComposer.web;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.ActionMessage;
import org.apache.struts.action.ActionMessages;
import org.apache.struts.util.MessageResources;
import org.lamsfoundation.lams.tool.ToolAccessMode;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerApplicationException;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerConstants;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerUser;
import org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService;
import org.lamsfoundation.lams.tool.behaviourComposer.service.BehaviourComposerServiceProxy;
import org.lamsfoundation.lams.tool.behaviourComposer.util.BehaviourComposerWebUtil;
import org.lamsfoundation.lams.usermanagement.dto.UserDTO;
import org.lamsfoundation.lams.util.WebUtil;
import org.lamsfoundation.lams.web.action.LamsDispatchAction;
import org.lamsfoundation.lams.web.session.SessionManager;
import org.lamsfoundation.lams.web.util.AttributeNames;
/**
 * Creation Date: 27-06-05
 * 
 * The learner url can be of three modes learner, teacher or author. Depending on 
 * what mode was set, it will trigger the corresponding action. If the mode parameter
 * is missing or a key is not found in the keymap, it will call the unspecified method
 * which defaults to the learner action. 
 *  
 * <p>The learner action, checks the defineLater and runOffline flags, and if required
 * it will show the learner the message screen indicating a reason why they cant see
 * the contents of the behaviourComposer.
 * If none of the flags are set, then the learner is able to see the behaviourComposer. 
 * </p>
 * <p>The difference between author mode (which is basically the preview)
 * is that if the defineLater flag is set, it will not be able to see the behaviourComposer screen
 * </p>
 * 
 * ----------------XDoclet Tags--------------------
 * 
 * @struts:action path="/starter/learner" name="BehaviourComposerLearnerForm" scope="request" type="org.lamsfoundation.lams.tool.behaviourComposer.web.BehaviourComposerLearnerStarterAction"
 *               validate="false" parameter="mode"
 * @struts:action-forward name="displayLearnerContent" path=".learnerContent"
 * @struts:action-forward name="displayMessage" path=".message"
 * @struts:action-forward name="defineLater" path=".defineLater"
 * ----------------XDoclet Tags--------------------
 */

public class BehaviourComposerLearnerStarterAction extends LamsDispatchAction {
    
    static Logger logger = Logger.getLogger(BehaviourComposerLearnerStarterAction.class.getName());
   
    /** Get the user id from the shared session */
	public Long getUserID(HttpServletRequest request) {
		// set up the user details
		HttpSession ss = SessionManager.getSession();
		UserDTO user = (UserDTO) ss.getAttribute(AttributeNames.USER);
		if ( user == null )
		{
		    MessageResources resources = getResources(request);
		    String error = resources.getMessage(BehaviourComposerConstants.ERR_MISSING_PARAM, "User");
		    logger.error(error);
			throw new BehaviourComposerApplicationException(error);
		}
        return new Long(user.getUserID().longValue());
	}

    public ActionForward unspecified(
    		ActionMapping mapping,
    		ActionForm form,
    		HttpServletRequest request,
    		HttpServletResponse response)
    {
        
        return learner(mapping, form, request, response);
    }
   
    public ActionForward learner(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws BehaviourComposerApplicationException {

        BehaviourComposerContent behaviourComposerContent = null;
        BehaviourComposerUser behaviourComposerUser = null;
        saveMessages(request, null);
        
        BehaviourComposerLearnerForm learnerForm = (BehaviourComposerLearnerForm)form;
      
        ActionMessages message = new ActionMessages();
        IBehaviourComposerService behaviourComposerService = BehaviourComposerServiceProxy.getBehaviourComposerService(getServlet().getServletContext());
        
        Long userID = getUserID(request);
        Long toolSessionID = BehaviourComposerWebUtil.convertToLong(learnerForm.getToolSessionID());
        
        if (toolSessionID == null)
		{
        	String error = "Unable to continue. The parameters tool session id is missing";
		    logger.error(error);
		    throw new BehaviourComposerApplicationException(error);
		}

        behaviourComposerContent = behaviourComposerService.retrieveBehaviourComposerBySessionID(toolSessionID);
	       // behaviourComposerSession = behaviourComposerService.retrieveBehaviourComposerSession(toolSessionId);
	   
	    
	    if(behaviourComposerContent == null)
		{
			String error = "An Internal error has occurred. Please exit and retry this sequence";
			logger.error(error);				
			throw new BehaviourComposerApplicationException(error);
		}   

	    behaviourComposerUser = behaviourComposerService.retrieveBehaviourComposerUserBySession(userID, toolSessionID);
	    
	    if ( isFlagSet(behaviourComposerContent, BehaviourComposerConstants.FLAG_DEFINE_LATER) ) {
            return mapping.findForward(BehaviourComposerConstants.DEFINE_LATER);
	    }

	    /*
         * Checks to see if the runOffline flag is set.
         * If the particular flag is set, control is forwarded to jsp page
         * displaying to the user the message according to what flag is set.
         */
        if (displayMessageToUser(behaviourComposerContent, message))
        {
            saveMessages(request, message);
            return mapping.findForward(BehaviourComposerConstants.DISPLAY_MESSAGE);
        }
        
        boolean readOnly = false;
        ToolAccessMode mode = WebUtil.readToolAccessModeParam(request, AttributeNames.PARAM_MODE,false);
        if (mode == ToolAccessMode.LEARNER || mode == ToolAccessMode.AUTHOR )
        {
        	if ( ! behaviourComposerContent.isContentInUse() ) {
	            /* Set the ContentInUse flag to true, and defineLater flag to false */
	            behaviourComposerContent.setContentInUse(true);
	            behaviourComposerService.saveBehaviourComposer(behaviourComposerContent);
        	}
        	
            if (behaviourComposerUser != null)
	        {
            	if (behaviourComposerUser.getUserStatus().equals(BehaviourComposerUser.COMPLETED)) {
            		readOnly = true;
            	}
	        }
            else
            {
            	//create a new user with this session id
            	BehaviourComposerUser newUser = new BehaviourComposerUser(userID);
            	behaviourComposerService.addUser(toolSessionID, newUser);
            }
        } else {
    		readOnly = true;
        }
        
        learnerForm.copyValuesIntoForm(behaviourComposerContent, readOnly, mode.toString());
        return mapping.findForward(BehaviourComposerConstants.DISPLAY_LEARNER_CONTENT);
    
    }
    
    public ActionForward teacher(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws BehaviourComposerApplicationException {
        return learner(mapping, form, request, response);
    }
    
    public ActionForward author(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws BehaviourComposerApplicationException {
        
    	return learner(mapping, form, request, response);

    }
    
    
    
    
    /**
     * <p>Performs a check on the flag indicated by <code>flag</code>
     * In this behaviourComposer tool, there are 3 possible flags:
     * <li>defineLater</li>
     * <li>contentInUse</li>
     * <li>runOffline</li>
     * <br>Returns true if the flag is set, false otherwise <p>
     * 
     * @param content The instance of BehaviourComposerContent
     * @param flag The flag to check, can take the following set of values (defineLater, contentInUse, runOffline)
     * @return Returns true if flag is set, false otherwise
     */
	private boolean isFlagSet(BehaviourComposerContent content, int flag) throws BehaviourComposerApplicationException
	{
	    switch (flag)
	    {
	    	case BehaviourComposerConstants.FLAG_DEFINE_LATER:
	    	    return (content.isDefineLater());
	    	  //  break;
	    	case BehaviourComposerConstants.FLAG_CONTENT_IN_USE:
	    		return (content.isContentInUse());
	    	//	break;
	    	case BehaviourComposerConstants.FLAG_RUN_OFFLINE:
	    		return(content.isForceOffline()); 
	    //		break;
	    	default:
	    	    throw new BehaviourComposerApplicationException("Invalid flag");
	    }
	  
	}
	
	/**
	 * <p> This methods checks the defineLater and runOffline flag. 
	 * If defineLater flag is set, then a message is added to an ActionMessages
	 * object saying that the contents have not been defined yet. If the runOffline 
	 * flag is set, a message is added to ActionMessages saying that the contents
	 * are not being run online. 
	 * This method will return true if any one of the defineLater or runOffline flag is set.
	 * Otherwise false will be returned. </p>
	 * 
	 * @param content The instance of BehaviourComposerContent
	 * @param message the instance of ActtionMessages
	 * @return true if any of the flags are set, false otherwise
	 */
	private boolean displayMessageToUser(BehaviourComposerContent content, ActionMessages message)
	{
	    boolean isDefineLaterSet = isFlagSet(content, BehaviourComposerConstants.FLAG_DEFINE_LATER);
        boolean isRunOfflineSet = isFlagSet(content, BehaviourComposerConstants.FLAG_RUN_OFFLINE);
        if(isDefineLaterSet || isRunOfflineSet)
        {
            if (isDefineLaterSet)
            {
                message.add(ActionMessages.GLOBAL_MESSAGE, new ActionMessage("message.defineLaterSet"));
            }
            if (isRunOfflineSet)
            {
                message.add(ActionMessages.GLOBAL_MESSAGE, new ActionMessage("message.runOfflineSet"));
            }
            return true;
        }
        else
            return false;
	}
	
	
 }
