/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: IBehaviourComposerService.java,v 1.16 2006/09/07 00:18:32 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.behaviourComposer.service;

import java.io.InputStream;
import java.util.List;

import org.lamsfoundation.lams.contentrepository.NodeKey;
import org.lamsfoundation.lams.contentrepository.RepositoryCheckedException;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerAttachment;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerUser;


/**
 * Defines the contract that the tool service provider must follow
 * 
 * @author mtruong
 */
public interface IBehaviourComposerService {
	
    //===================================================================
    // BehaviourComposerContent access methods
    //===================================================================	
    
  
    
    /**
     * <p> Returns the persistent instance of BehaviourComposerContent
     * with the given tool session id <code>behaviourComposerSessionId</code>, returns null if not found.
     * 
     * @param behaviourComposerSessionId The tool session id
     * @return a persistent instance of BehaviourComposerContent or null if not found.
     */	
	public BehaviourComposerContent retrieveBehaviourComposerBySessionID(Long behaviourComposerSessionId);
    
    /**
	 * <p>Retrieve an instance of BehaviourComposerContent with the given
     * tool content id <code>behaviourComposerContentId</code> </p>
     * @param behaviourComposerContentId The tool content id
     * @return an instance of BehaviourComposerContent
     */
	public BehaviourComposerContent retrieveBehaviourComposer(Long behaviourComposerContentId);
	
	
	/**
	 * <p>Persist/Update the given persistent instance of BehaviourComposerContent.</p>
	 * 
	 * @param behaviourComposerContent The instance of BehaviourComposerContent to persist.
	 */
	public void saveBehaviourComposer(BehaviourComposerContent behaviourComposerContent);
	
	
	 /**
     * <p>Deletes all instances of BehaviourComposerSession that are associated
     * with the given instance of BehaviourComposerContent</p>
     * 
     * @param behaviourComposerContent The instance of BehaviourComposerContent in which corresponding instances of BehaviourComposerSession should be deleted.
     */
	public void removeBehaviourComposerSessionsFromContent(BehaviourComposerContent behaviourComposerContent);
	
	/**
     * <p>Delete the given instance of BehaviourComposerContent with the
     * given tool content id <code>behaviourComposerContentId</code>
     * 
     * @param behaviourComposerContentId The tool content Id. 
     */
	public void removeBehaviourComposer(Long behaviourComposerContentId);
	
	/**
     * <p>Delete the given instance of BehaviourComposerContent</p>
     * 
     * @param behaviourComposerContent The instance of BehaviourComposerContent to delete. 
     */
    public void removeBehaviourComposer(BehaviourComposerContent behaviourComposerContent);
	
	
    //===================================================================
    // BehaviourComposerSession access methods
    //===================================================================
    /**
	 * <p> Return the persistent instance of a BehaviourComposerSession
	 * with the given tool session id <code>behaviourComposerSessionId</code>,
	 * returns null if not found.</p>
	 * 
	 * @param behaviourComposerSessionId The tool session id
	 * @return the persistent instance of a BehaviourComposerSession or null if not found.
	 */
	public BehaviourComposerSession retrieveBehaviourComposerSession(Long behaviourComposerSessionId);
	

	/**
	 * Persists the new BehaviourComposerSession object into the database.
	 * 
	 * @param behaviourComposerSession the BehaviourComposerSession object to persist
	 */
	public void saveBehaviourComposerSession(BehaviourComposerSession behaviourComposerSession);
	
	
	/**
	 * Updates the values of the behaviourComposer session.
	 * @param behaviourComposerSession
	 */
	public void updateBehaviourComposerSession(BehaviourComposerSession behaviourComposerSession);
	
	/**
	 * Remove the behaviourComposer session object with session id of
	 * that specified in the argument.
	 * 
	 * @param behaviourComposerSessionId The id of the requested behaviourComposer object
	 *
	 */	
	public void removeSession(Long behaviourComposerSessionId);
	
	 /**
     * <p>Delete the given instance of BehaviourComposerSession</p>
     * 
     * @param behaviourComposerSession The instance of BehaviourComposerSession to delete. 
     */
    public void removeSession(BehaviourComposerSession behaviourComposerSession);
    
     
    /**
     * <p>Deletes all instances of BehaviourComposerUser that are associated
     * with the given instance of BehaviourComposerSession</p>
     * 
     * @param behaviourComposerSession The instance of BehaviourComposerSession in which corresponding instances of BehaviourComposerUser should be deleted.
     */
	public void removeBehaviourComposerUsersFromSession(BehaviourComposerSession behaviourComposerSession);
	
	  /**
     * <p> Returns the persistent instance of BehaviourComposerSession
     * with the given behaviourComposer user id<code>userId</code>, returns null if not found.
     * 
     * @param userId The user id
     * @return a persistent instance of BehaviourComposerSession or null if not found.
     */	
	public BehaviourComposerSession retrieveBehaviourComposerSessionByUserID(Long userId);
    
    //===================================================================
    // BehaviourComposerUser access methods
    //===================================================================
    /**
	 * <p> Return the persistent instance of a BehaviourComposerUser
	 * with the given user id<code>behaviourComposerUserId</code>,
	 * returns null if not found.</p>
	 * 
	 * @param behaviourComposerUserId The user id of the instance of BehaviourComposerUser
	 * @return the persistent instance of a BehaviourComposerUser or null if not found.
	 */
	public BehaviourComposerUser retrieveBehaviourComposerUser(Long behaviourComposerUserId);
	
	/**
	 * <p> Return the persistent instance of a BehaviourComposerUser
	 * who has the user id <code>userId</code> and tool session id
	 * <code>sessionId</code>
	 * returns null if not found.</p>
	 * 
	 * @param userId. The id of the learner
	 * @param sessionId. The tool session id to which this user belongs to.
	 * @return the persistent instance of a BehaviourComposerUser or null if not found.
	 */
	public BehaviourComposerUser retrieveBehaviourComposerUserBySession(Long userId, Long sessionId);
	
	/**
	 * Persists the new BehaviourComposerUser object into the database.
	 * 
	 * @param behaviourComposerUser the BehaviourComposerUser object to persist
	 */
	public void saveBehaviourComposerUser(BehaviourComposerUser behaviourComposerUser);
	
	
	/**
	 * Updates the values of the behaviourComposer user.
	 * @param behaviourComposerUser
	 */
	public void updateBehaviourComposerUser(BehaviourComposerUser behaviourComposerUser);
	
	/**
	 * Remove the behaviourComposer user object with user id of
	 * that specified in the argument.
	 * 
	 * @param behaviourComposerUserId The id of the requested behaviourComposer object
	 *
	 */	
	public void removeUser(Long behaviourComposerUserId);
	
	 /**
     * <p>Delete the given instance of BehaviourComposerUser</p>
     * 
     * @param behaviourComposerUser The instance of BehaviourComposerUser to delete. 
     */
    public void removeUser(BehaviourComposerUser behaviourComposerUser);
 
    /**
     * <p> Saves the instance of BehaviourComposerSession to the database. 
     * This instance is added to the collection of sessions from 
     * BehaviourComposerContent with tool content id <code>behaviourComposerContentId</code> </p>
     * 
     * @param behaviourComposerContentId The tool content Id
     * @param session The instance of BehaviourComposerSession to persist
     */
    public void addSession(Long behaviourComposerContentId, BehaviourComposerSession session);
    
    /**
     * <p>Saves the instance of BehaviourComposerUser to the database.
     * <code>behaviourComposerUser</code> is added to the collection of users from 
     * BehaviourComposerSession with tool session id <code>behaviourComposerSessionId</code> </p>
     * 
     * @param behaviourComposerSessionId The tool session id
     * @param user The instance of BehaviourComposerUser to persist
     */
    public void addUser(Long behaviourComposerSessionId, BehaviourComposerUser behaviourComposerUser);
    
    /**
     * <p>Retrieves a list of the session IDs from the given instance of BehaviourComposerContent</p> 
     * @param content 
     * @return list of session ids (Long)
     */
    public List getSessionIdsFromContent(BehaviourComposerContent content);
    
    /**
     * <p>Returns the number of users in this session</p>
     * @param session
     * @return The number of users in the session
     */
    public int getNumberOfUsersInSession(BehaviourComposerSession session);
    
    /**
     * <p>Finds the number of learners that have participated in this tool activity
     * with the given toolContentId. It finds all the toolSessionIds relating to this
     * toolContentId, and calculates the number of users in each tool session(group).
     * Returns the total number of users across all sessions </p>
     * @param toolContentId 
     * @return the total number of users for this tool activity
     */
    public int calculateTotalNumberOfUsers(Long toolContentId);
    
    //===================================================================
    // BehaviourComposerAttachment access methods
    //===================================================================
    
    /**
	 * <p>Retrieve an instance of BehaviourComposerAttachment with the given
     * attachment id <code>attachmentId</code> </p>
     * @param attachmentId The id for the attachment file
     * @return an instance of BehaviourComposerAttachment
     */
    public BehaviourComposerAttachment retrieveAttachment(Long attachmentId);
    
    /**
	 * <p>Retrieve the file attachment with the given uuid </p>
     * @param uuid The unique identifier for the file, corresponds to the uuid for the file stored in content repository
     * @return an instance of BehaviourComposerAttachment
     */
    public BehaviourComposerAttachment retrieveAttachmentByUuid(Long uuid);
    
    /**
	 * <p>Retrieve an instance of BehaviourComposerAttachment with the 
     * filename <code>filename</code> </p>
     * @param filename The filename of the attachment that you want to retrieve
     * @return an instance of BehaviourComposerAttachment
     */
    public BehaviourComposerAttachment retrieveAttachmentByFilename(String filename);
    
    /**
     * <p>Retrieve the list of attachment ids with the given instance of BehaviourComposerContent</p>
     * @param behaviourComposerContent The given instance of BehaviourComposerContent
     * @return List. the list of attachment ids (java.lang.Long)
     */
    public List getAttachmentIdsFromContent(BehaviourComposerContent behaviourComposerContent);
    
    /**
     * <p> Saves (persists) or update the BehaviourComposerAttachment object in the
     * database.</p>
     * @param content The overall behaviourComposer content object to which the attachment is to be added
     * @param attachment The instance of BehaviourComposerAttachment to save
     */
    public void saveAttachment(BehaviourComposerContent content, BehaviourComposerAttachment attachment);
    
    /**
     * Removes the BehaviourComposerAttachment object from the database.
     * @param content The overall behaviourComposer content object to which the attachment is to be added
     * @param attachment The instance of BehaviourComposerAttachment to delete.
     */
    public void removeAttachment(BehaviourComposerContent content, BehaviourComposerAttachment attachment) throws RepositoryCheckedException;
    
	/** 
	 * Add a file to the content repository. Does not add a record to the behaviourComposer tables.
	 * @throws RepositoryCheckedException 
	 */
	public NodeKey uploadFile(InputStream istream, String filename, String contentType, String fileType) throws RepositoryCheckedException;

    /**
     * This method retrieves the default content id.
     * @param toolSignature The tool signature which is defined in lams_tool table.
     * @return the default content id
     */
    public Long getToolDefaultContentIdBySignature(String toolSignature);
    
}
