/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: BehaviourComposerServicePOJO.java,v 1.32 2006/09/07 00:18:32 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.behaviourComposer.service;

import java.io.InputStream;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.lamsfoundation.lams.contentrepository.ItemNotFoundException;
import org.lamsfoundation.lams.contentrepository.NodeKey;
import org.lamsfoundation.lams.contentrepository.RepositoryCheckedException;
import org.lamsfoundation.lams.contentrepository.client.IToolContentHandler;
import org.lamsfoundation.lams.learning.service.ILearnerService;
import org.lamsfoundation.lams.learningdesign.service.ExportToolContentException;
import org.lamsfoundation.lams.learningdesign.service.IExportToolContentService;
import org.lamsfoundation.lams.learningdesign.service.ImportToolContentException;
import org.lamsfoundation.lams.tool.ToolContentImport102Manager;
import org.lamsfoundation.lams.tool.ToolContentManager;
import org.lamsfoundation.lams.tool.ToolSessionExportOutputData;
import org.lamsfoundation.lams.tool.ToolSessionManager;
import org.lamsfoundation.lams.tool.exception.DataMissingException;
import org.lamsfoundation.lams.tool.exception.SessionDataExistsException;
import org.lamsfoundation.lams.tool.exception.ToolException;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerApplicationException;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerAttachment;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerConstants;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerUser;
import org.lamsfoundation.lams.tool.behaviourComposer.dao.IBehaviourComposerAttachmentDAO;
import org.lamsfoundation.lams.tool.behaviourComposer.dao.IBehaviourComposerContentDAO;
import org.lamsfoundation.lams.tool.behaviourComposer.dao.IBehaviourComposerSessionDAO;
import org.lamsfoundation.lams.tool.behaviourComposer.dao.IBehaviourComposerUserDAO;
import org.lamsfoundation.lams.tool.service.ILamsToolService;
import org.lamsfoundation.lams.usermanagement.dto.UserDTO;
import org.springframework.dao.DataAccessException;



/**
 * An implementation of the BehaviourComposerService interface.
 * 
 * As a requirement, all LAMS tool's service bean must implement ToolContentManager and ToolSessionManager.
 * @author mtruong
 *
 */
public class BehaviourComposerServicePOJO implements IBehaviourComposerService, ToolContentManager, ToolSessionManager, ToolContentImport102Manager {

	private BehaviourComposerContent behaviourComposerContent;
	private IBehaviourComposerContentDAO behaviourComposerContentDAO=null;
	
	private BehaviourComposerSession behaviourComposerSession;
	private IBehaviourComposerSessionDAO behaviourComposerSessionDAO = null;
	
	private ILearnerService learnerService;
	private ILamsToolService toolService;
	
	private BehaviourComposerUser behaviourComposerUser;
	private IBehaviourComposerUserDAO behaviourComposerUserDAO=null;
	
	private IBehaviourComposerAttachmentDAO behaviourComposerAttachmentDAO = null;
	private IToolContentHandler behaviourComposerToolContentHandler = null;
	
	private IExportToolContentService exportContentService;
	private static Logger log = Logger.getLogger(BehaviourComposerServicePOJO.class);

	
	/* ==============================================================================
	 * Methods for access to BehaviourComposerContent objects
	 * ==============================================================================
	 */
	
	
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveBehaviourComposer(Long)
	 */
	public BehaviourComposerContent retrieveBehaviourComposer(Long behaviourComposerContentId) throws BehaviourComposerApplicationException
	{
	    if (behaviourComposerContentId == null)
	    {
	        String error = "Unable to continue. The tool content id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	       
		try
		{
			behaviourComposerContent = behaviourComposerContentDAO.findBehaviourComposerContentById(behaviourComposerContentId);
						
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("An exception has occured when trying to retrieve behaviourComposer content: "
                                                         + e.getMessage(),
														   e);
		}
		
		return behaviourComposerContent;
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveBehaviourComposerBySessionID(Long)
	 */
	public BehaviourComposerContent retrieveBehaviourComposerBySessionID(Long behaviourComposerSessionId)
	{
	    if (behaviourComposerSessionId == null)
	    {
	        String error = "Unable to continue. The tool session id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
	    try
		{
			behaviourComposerContent = behaviourComposerContentDAO.getBehaviourComposerContentBySession(behaviourComposerSessionId);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("An exception has occured when trying to retrieve behaviourComposer content: "
                                                         + e.getMessage(),
														   e);
		}
		
		return behaviourComposerContent;
	}
	
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#saveBehaviourComposer(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent)
	 */
	public void saveBehaviourComposer(BehaviourComposerContent behaviourComposerContent)
	{
		try
		{
			if ( behaviourComposerContent.getUid() == null ) {
				behaviourComposerContentDAO.saveBehaviourComposerContent(behaviourComposerContent);
			} else {
				behaviourComposerContentDAO.updateBehaviourComposerContent(behaviourComposerContent);
			}
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to save the behaviourComposer content object: "
														+ e.getMessage(), e);
		}
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeBehaviourComposerSessions(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent)
	 */
	public void removeBehaviourComposerSessionsFromContent(BehaviourComposerContent behaviourComposerContent)
	{
		try
		{
		    behaviourComposerContent.getBehaviourComposerSessions().clear();
		    //updateBehaviourComposer(behaviourComposerContent);
		    
			behaviourComposerContentDAO.removeBehaviourComposerSessions(behaviourComposerContent);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove the sessions associated with this behaviourComposer content object: "
														+ e.getMessage(), e);
		}
		
	}
	 
	/**
     * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeBehaviourComposer(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent)
     */
	public void removeBehaviourComposer(Long behaviourComposerContentId)
	{
	    if (behaviourComposerContentId == null)
	    {
	        String error = "Unable to continue. The tool content id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
		try
		{
			behaviourComposerContentDAO.removeBehaviourComposer(behaviourComposerContentId);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove this behaviourComposer content object: "
					+ e.getMessage(), e);
		}
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeBehaviourComposer(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent)
	 */
	public void removeBehaviourComposer(BehaviourComposerContent behaviourComposerContent)
	{
	    try
		{
			behaviourComposerContentDAO.removeBehaviourComposer(behaviourComposerContent);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove this behaviourComposer content object: "
					+ e.getMessage(), e);
		}
	}
	
		
	/* ==============================================================================
	 * Methods for access to BehaviourComposerSession objects
	 * ==============================================================================
	 */
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveBehaviourComposerSession(Long)
	 */
	public BehaviourComposerSession retrieveBehaviourComposerSession(Long behaviourComposerSessionId)
	{
	    if (behaviourComposerSessionId == null)
	    {
	        String error = "Unable to continue. The tool session id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
	    try
	    {
	        behaviourComposerSession = behaviourComposerSessionDAO.findBehaviourComposerSessionById(behaviourComposerSessionId);
	    }
	    catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("An exception has occured when trying to retrieve behaviourComposer session object: "
                                                         + e.getMessage(),
														   e);
		}
		
		return behaviourComposerSession;
	}
	
	
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#saveBehaviourComposerSession(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession)
	 */
	public void saveBehaviourComposerSession(BehaviourComposerSession behaviourComposerSession)
	{
	    try
	    {
	        BehaviourComposerContent content = behaviourComposerSession.getBehaviourComposerContent();
	     //   content.getBehaviourComposerSessions().add(behaviourComposerSession);
	       // content.
	        
	     /*   updateBehaviourComposer(content); */
	        behaviourComposerSessionDAO.saveBehaviourComposerSession(behaviourComposerSession);
	    }
	    catch(DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to save this behaviourComposer session: "
	                	+e.getMessage(), e);
	    }
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#updateBehaviourComposerSession(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession)
	 */
	public void updateBehaviourComposerSession(BehaviourComposerSession behaviourComposerSession)
	{
	    try
	    {
	        behaviourComposerSessionDAO.updateBehaviourComposerSession(behaviourComposerSession);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to update this behaviourComposer session: "
                	+e.getMessage(), e);
	    }
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeSession(Long)
	 */
	public void removeSession(Long behaviourComposerSessionId)
	{
	    if (behaviourComposerSessionId == null)
	    {
	        String error = "Unable to continue. The tool session id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
	    try
		{
	        BehaviourComposerSession sessionToDelete = retrieveBehaviourComposerSession(behaviourComposerSessionId);
	        BehaviourComposerContent contentReferredBySession = sessionToDelete.getBehaviourComposerContent();
	        //un-associate the session from content
	        contentReferredBySession.getBehaviourComposerSessions().remove(sessionToDelete);
	        behaviourComposerSessionDAO.removeBehaviourComposerSession(behaviourComposerSessionId);
	      //  updateBehaviourComposer(contentReferredBySession);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove this behaviourComposer session object: "
					+ e.getMessage(), e);
		}
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeSession(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession)
	 */
	public void removeSession(BehaviourComposerSession behaviourComposerSession)
	{
	    try
		{
	        BehaviourComposerContent contentReferredBySession = behaviourComposerSession.getBehaviourComposerContent();
	        //un-associate the session from content
	        contentReferredBySession.getBehaviourComposerSessions().remove(behaviourComposerSession);
	              
			behaviourComposerSessionDAO.removeBehaviourComposerSession(behaviourComposerSession);
		//	updateBehaviourComposer(contentReferredBySession);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove this behaviourComposer session object: "
					+ e.getMessage(), e);
		}
	}
	
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeBehaviourComposerUsersFromSession(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession)
	 */
	public void removeBehaviourComposerUsersFromSession(BehaviourComposerSession behaviourComposerSession)
	{
		try
		{
		    behaviourComposerSession.getBehaviourComposerUsers().clear();
		//    updateBehaviourComposerSession(behaviourComposerSession);
		    
			behaviourComposerSessionDAO.removeBehaviourComposerUsers(behaviourComposerSession);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove the users associated with this behaviourComposer session instance: "
														+ e.getMessage(), e);
		}
		
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveBehaviourComposerSessionByUserID(java.lang.Long)
	 */
	public BehaviourComposerSession retrieveBehaviourComposerSessionByUserID(Long userId)
	{
	    if (userId == null)
	    {
	        String error = "Unable to continue. The tool session id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
		try
		{
			behaviourComposerSession = behaviourComposerSessionDAO.getBehaviourComposerSessionByUser(userId);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to retrieve behaviourComposer session instance "
														+ e.getMessage(), e);
		}
		return behaviourComposerSession;
		
	}
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#getSessionIdsFromContent(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent) */
	public List getSessionIdsFromContent(BehaviourComposerContent content)
	{
	    List list = null;
	    try
	    {
	        list = behaviourComposerSessionDAO.getSessionsFromContent(content);
	    }
	    catch(DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to the list of session ids from content "
					+ e.getMessage(), e);
	    }
	    return list;
	}
	
	/* ==============================================================================
	 * Methods for access to BehaviourComposerUser objects
	 * ==============================================================================
	 */
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveBehaviourComposerUser(java.lang.Long)
	 */
	public BehaviourComposerUser retrieveBehaviourComposerUser(Long behaviourComposerUserId)
	{
	    if (behaviourComposerUserId == null)
	    {
	        String error = "Unable to continue. The user id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
	    try
		{
			behaviourComposerUser = behaviourComposerUserDAO.getBehaviourComposerUserByID(behaviourComposerUserId);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("An exception has occured when trying to retrieve behaviourComposer user: "
                                                         + e.getMessage(),
														   e);
		}
		
		return behaviourComposerUser;
	}
	
	
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#saveBehaviourComposerUser(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerUser)
	 */
	public void saveBehaviourComposerUser(BehaviourComposerUser behaviourComposerUser)
	{
		try
		{
		    BehaviourComposerSession session = behaviourComposerUser.getBehaviourComposerSession();
		    session.getBehaviourComposerUsers().add(behaviourComposerUser);
		  //  updateBehaviourComposerSession(session);
		    
			behaviourComposerUserDAO.saveBehaviourComposerUser(behaviourComposerUser);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to save the behaviourComposer user object: "
														+ e.getMessage(), e);
		}
	}
	
	/** org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveBehaviourComposerUserBySession(java.lang.Long, java.lang.Long) */
	public BehaviourComposerUser retrieveBehaviourComposerUserBySession(Long userId, Long sessionId)
	{
		try
		{
		  behaviourComposerUser = behaviourComposerUserDAO.getBehaviourComposerUserBySession(userId, sessionId);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to retrive the behaviourComposer user object: "
					+ e.getMessage(), e);
		}
		
		return behaviourComposerUser;
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#updateBehaviourComposerUser(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerUser)
	 */
	public void updateBehaviourComposerUser(BehaviourComposerUser behaviourComposerUser)
	{
		try
		{
			behaviourComposerUserDAO.updateBehaviourComposerUser(behaviourComposerUser);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to update the behaviourComposer user object: "
														+ e.getMessage(), e);
		}
	}
	
		
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeUser(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerUser)
	 */
	public void removeUser(BehaviourComposerUser behaviourComposerUser)
	{
		try
		{
		    BehaviourComposerSession session = behaviourComposerUser.getBehaviourComposerSession();
		    session.getBehaviourComposerUsers().remove(behaviourComposerUser);
		    
			behaviourComposerUserDAO.removeBehaviourComposerUser(behaviourComposerUser);
			
		//	updateBehaviourComposerSession(session);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove the behaviourComposer user object: "
														+ e.getMessage(), e);
		}
	}
	
	/**
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeUser(java.lang.Long)
	 */
	public void removeUser(Long behaviourComposerUserId)
	{
	    if (behaviourComposerUserId == null)
	    {
	        String error = "Unable to continue. The user id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
		try
		{
		    BehaviourComposerUser user = retrieveBehaviourComposerUser(behaviourComposerUserId);
		    BehaviourComposerSession session = user.getBehaviourComposerSession();
		    session.getBehaviourComposerUsers().remove(user);
			behaviourComposerUserDAO.removeBehaviourComposerUser(behaviourComposerUserId);
			
		//	updateBehaviourComposerSession(session);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove the behaviourComposer user object: "
														+ e.getMessage(), e);
		}
	}
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#addSession(java.lang.Long, org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession) */
	public void addSession(Long behaviourComposerContentId, BehaviourComposerSession session)
    {

	    if (behaviourComposerContentId == null || session == null)
	    {
	        String error = "Unable to continue. The tool content id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
        try
		{
		    behaviourComposerContentDAO.addBehaviourComposerSession(behaviourComposerContentId, session);
		}
		catch (DataAccessException e)
		{
			throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to create session: "
														+ e.getMessage(), e);
		}
    }	
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#addUser(java.lang.Long, org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession) */
	public void addUser(Long behaviourComposerSessionId, BehaviourComposerUser user)
	{

	    if (behaviourComposerSessionId == null)
	    {
	        String error = "Unable to continue. The tool session id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    } 
	    try
	    {
	        behaviourComposerSessionDAO.addBehaviourComposerUsers(behaviourComposerSessionId, user);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to create user: "
					+ e.getMessage(), e);
	    }
	}
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#getNumberOfUsersInSession(org.lamsfoundation.lams.tool.behaviourComposer.oticeboardSession) */
	public int getNumberOfUsersInSession(BehaviourComposerSession session)
	{
	    int numberOfUsers;
	    try
	    {
	        numberOfUsers = behaviourComposerUserDAO.getNumberOfUsers(session);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to get the number of users in the session: "
					+ e.getMessage(), e);
	    }
	    return numberOfUsers;
	}
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#calculateTotalNumberOfUsers(java.lang.Long) */
	public int calculateTotalNumberOfUsers(Long toolContentId)
	{

	    if (toolContentId == null)
	    {
	        String error = "Unable to continue. The tool content id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
	    int totalNumberOfUsers = 0;
	    try
	    {
	        behaviourComposerContent = retrieveBehaviourComposer(toolContentId);
	        List listOfSessionIds = getSessionIdsFromContent(behaviourComposerContent);
	        
	        Iterator i = listOfSessionIds.iterator();
	        
	        while(i.hasNext())
	        {
	            Long sessionId = (Long)i.next();
	            int usersInThisSession = getNumberOfUsersInSession(retrieveBehaviourComposerSession(sessionId));
	            totalNumberOfUsers = totalNumberOfUsers + usersInThisSession;
	        }
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while calculating the total number of users in tool activity "
					+ e.getMessage(), e);
	    }
	    return totalNumberOfUsers;
	}
	
	/* ==============================================================================
	 * Methods for access to BehaviourComposerUser objects
	 * ==============================================================================
	 */
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveAttachment(java.lang.Long) */
	public BehaviourComposerAttachment retrieveAttachment(Long attachmentId)
	{
	    if (attachmentId == null)
	    {
	        String error = "Unable to continue. The attachment id is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    
	    try
	    {
	        return behaviourComposerAttachmentDAO.retrieveAttachment(attachmentId);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to retrieve the attachment "
	                + e.getMessage(), e);
	    }
	}
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveAttachmentByUuid(java.lang.Long) */
	public BehaviourComposerAttachment retrieveAttachmentByUuid(Long uuid)
	{
	    if (uuid == null)
	    {
	        String error = "Unable to continue. The uuid is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    try
	    {
	        return behaviourComposerAttachmentDAO.retrieveAttachmentByUuid(uuid);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to retrieve the attachment "
	                + e.getMessage(), e);
	    }
	}
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#retrieveAttachment(java.lang.String) */
	public BehaviourComposerAttachment retrieveAttachmentByFilename(String filename)
	{
	    if (filename == null || filename.trim().length() == 0)
	    {
	        String error = "Unable to continue. The filename is missing";
	        log.error(error);
	        throw new BehaviourComposerApplicationException(error);
	    }
	    try
	    {
	        return behaviourComposerAttachmentDAO.retrieveAttachmentByFilename(filename);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to retrieve the attachment with filename " + filename + " "
	                + e.getMessage(), e);
	    }
	}
	
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#getAttachmentIdsFromContent(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent) */
	public List getAttachmentIdsFromContent(BehaviourComposerContent behaviourComposerContent)
	{
	    try
	    {
	        return behaviourComposerAttachmentDAO.getAttachmentIdsFromContent(behaviourComposerContent);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to retrieve the list of attachment ids "
	                + e.getMessage(), e);
	    }
	}
	
	/** @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#saveAttachment(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerAttachment) */
	public void saveAttachment(BehaviourComposerContent content, BehaviourComposerAttachment attachment)
	{
	    try
	    {
	    	content.getBehaviourComposerAttachments().add(attachment);
			attachment.setBehaviourComposerContent(content);
			saveBehaviourComposer(content);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to save the attachment "
	                + e.getMessage(), e);
	    }
	}
	
	/** @throws RepositoryCheckedException 
	 * @throws  
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#removeAttachment(org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerAttachment) */
	public void removeAttachment(BehaviourComposerContent content, BehaviourComposerAttachment attachment) throws RepositoryCheckedException
	{
	    try
	    {
			attachment.setBehaviourComposerContent(null);
			content.getBehaviourComposerAttachments().remove(attachment);
			behaviourComposerToolContentHandler.deleteFile(attachment.getUuid());
			saveBehaviourComposer(content);
	    }
	    catch (DataAccessException e)
	    {
	        throw new BehaviourComposerApplicationException("EXCEPTION: An exception has occurred while trying to remove this attachment"
	                + e.getMessage(), e);
	    }
	}
	
	/** @throws RepositoryCheckedException 
	 * @see org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService#uploadFile(java.io.InputStream, java.lang.String, java.lang.String, java.lang.String) */
	public NodeKey uploadFile(InputStream istream, String filename, String contentType, String fileType) throws RepositoryCheckedException
	{
	    return behaviourComposerToolContentHandler.uploadFile(istream, filename, contentType, fileType); 
	}
	
	/* ===============Methods implemented from ToolContentManager =============== */
	
	/** @see org.lamsfoundation.lams.tool.ToolContentManager#copyToolContent(java.lang.Long, java.lang.Long)*/
	public void copyToolContent(Long fromContentId, Long toContentId) throws ToolException {
	
	    if (toContentId == null)
		    throw new ToolException("Failed to copy BehaviourComposer tool content. Missing parameter: toContentId");
		if (fromContentId == null)
		{
		    //use the default content Id
		    //fromContentId = BehaviourComposerConstants.DEFAULT_CONTENT_ID;
		    fromContentId = getToolDefaultContentIdBySignature(BehaviourComposerConstants.TOOL_SIGNATURE);
		}
		
		//fromContentId might not have any content, in this case use default content
		//default content id might not have any contnet, throw exception
		BehaviourComposerContent originalBehaviourComposer = null;
		
		try {
			if ((originalBehaviourComposer = retrieveBehaviourComposer(fromContentId))== null) //the id given does not have content, use default content
			{
			    //use default content id to grab contents
			    BehaviourComposerContent defaultContent = retrieveBehaviourComposer(getToolDefaultContentIdBySignature(BehaviourComposerConstants.TOOL_SIGNATURE));
			    
			    if (defaultContent != null)
			    {
			        BehaviourComposerContent newContent = BehaviourComposerContent.newInstance(defaultContent, toContentId, behaviourComposerToolContentHandler);
			        saveBehaviourComposer(newContent);
			    }
			    else
			    {
			        throw new ToolException("Default content is missing. Unable to copy tool content");
			    }
			}
			else
			{
			    BehaviourComposerContent newBehaviourComposerContent = BehaviourComposerContent.newInstance(originalBehaviourComposer, toContentId, behaviourComposerToolContentHandler);
				saveBehaviourComposer(newBehaviourComposerContent);
			}
		} catch (RepositoryCheckedException e) {
			log.error("Unable to copy the tool content due to a content repository error. fromContentId "+fromContentId+" toContentId "+toContentId);
			throw new ToolException(e);
		}
			
		
	}
	
	/** @see org.lamsfoundation.lams.tool.ToolContentManager#setAsDefineLater(java.lang.Long)*/
	public void setAsDefineLater(Long toolContentId) throws DataMissingException, ToolException
	{
	    BehaviourComposerContent behaviourComposerContent = getAndCheckIDandObject(toolContentId);
		
	    behaviourComposerContent.setDefineLater(true);
	    //behaviourComposerContent.setContentInUse(false); //if define later is set to true, then contentInUse flag should be false
	    saveBehaviourComposer(behaviourComposerContent);
		
	}
	
	/** @see org.lamsfoundation.lams.tool.ToolContentManager#setAsRunOffline(java.lang.Long)*/
	public void setAsRunOffline(Long toolContentId) throws DataMissingException, ToolException
	{
	    BehaviourComposerContent behaviourComposerContent = getAndCheckIDandObject(toolContentId);
	    
		behaviourComposerContent.setForceOffline(true);
		saveBehaviourComposer(behaviourComposerContent);
	}
	   
	/** @see org.lamsfoundation.lams.tool.ToolContentManager#removeToolContent(java.lang.Long)*/
	public void removeToolContent(Long toolContentId, boolean removeSessionData) throws SessionDataExistsException, ToolException
	{	
	    BehaviourComposerContent behaviourComposerContent =  getAndCheckIDandObject(toolContentId);
	   //if session data exist and removeSessionData=false, throw an exception
	    if ((!behaviourComposerContent.getBehaviourComposerSessions().isEmpty()) && !removeSessionData)
	        throw new SessionDataExistsException("Delete failed: There is session data that belongs to this tool content id");
	    
	   //remove any attachments that belong to this tool entry
	   Set attachments = behaviourComposerContent.getBehaviourComposerAttachments();
	   Iterator i = attachments.iterator();
	   while(i.hasNext())
	   {
		   try
		   {
			   removeAttachment(behaviourComposerContent, (BehaviourComposerAttachment)i.next());
		   }
		   catch(RepositoryCheckedException e)
		   {
			   //TODO: not sure if suppose to throw another type of exception or not
		   }
	   }
	    
	   removeBehaviourComposer(toolContentId);
	}

	private BehaviourComposerContent getAndCheckIDandObject(Long toolContentId) throws ToolException, DataMissingException
	{
	    if (toolContentId == null)
		    throw new ToolException("Tool content ID is missing. Unable to continue");
	   
	    BehaviourComposerContent behaviourComposerContent = retrieveBehaviourComposer(toolContentId);
	    if (behaviourComposerContent == null)
	        throw new DataMissingException("No tool content matches this tool content id");
	    
	    return behaviourComposerContent;
	}
	
	private BehaviourComposerSession getAndCheckSessionIDandObject(Long toolSessionId) throws ToolException, DataMissingException
	{
	    if (toolSessionId == null)
		    throw new ToolException("Tool session ID is missing. Unable to continue");
	   
	    BehaviourComposerSession behaviourComposerSession = retrieveBehaviourComposerSession(toolSessionId);
	    if (behaviourComposerSession == null)
	        throw new DataMissingException("No tool session matches this tool session id");
	    
	    return behaviourComposerSession;
	} 
	
	/*private void checkSessionIDandObject(Long toolSessionId) throws ToolException, DataMissingException
	{
	    if (toolSessionId == null)
		    throw new ToolException("Tool session ID is missing. Unable to continue");
	   
	    BehaviourComposerSession behaviourComposerSession = retrieveBehaviourComposerSession(toolSessionId);
	    if (behaviourComposerSession == null)
	        throw new DataMissingException("No tool session matches this tool session id");
	} */

	/**
     * Export the XML fragment for the tool's content, along with any files needed
     * for the content.
     * @throws DataMissingException if no tool content matches the toolSessionId 
     * @throws ToolException if any other error occurs
     */

	public void exportToolContent(Long toolContentId, String rootPath) throws DataMissingException, ToolException {
		BehaviourComposerContent toolContentObj = behaviourComposerContentDAO.findBehaviourComposerContentById(toolContentId);
 		if(toolContentObj == null)
 			throw new DataMissingException("Unable to find tool content by given id :" + toolContentId);
 		
		try {
			//set ResourceToolContentHandler as null to avoid copy file node in repository again.
			toolContentObj = BehaviourComposerContent.newInstance(toolContentObj,toolContentId,null);
			toolContentObj.setBehaviourComposerSessions(null);
			exportContentService.registerFileClassForExport(BehaviourComposerAttachment.class.getName(),"uuid","versionId");
			exportContentService.exportToolContent( toolContentId, toolContentObj,behaviourComposerToolContentHandler, rootPath);
		} catch (ExportToolContentException e) {
			throw new ToolException(e);
		} catch (ItemNotFoundException e) {
			throw new ToolException(e);
		} catch (RepositoryCheckedException e) {
			throw new ToolException(e);
		}
	}

    /**
     * Import the XML fragment for the tool's content, along with any files needed
     * for the content.
     * @throws ToolException if any other error occurs
     */
	 public void importToolContent(Long toolContentId, Integer newUserUid, String toolContentPath) throws ToolException {
		 try {
				exportContentService.registerFileClassForImport(BehaviourComposerAttachment.class.getName()
						,"uuid","versionId","filename","fileProperty",null,null);
				
				Object toolPOJO =  exportContentService.importToolContent(toolContentPath,behaviourComposerToolContentHandler);
				if(!(toolPOJO instanceof BehaviourComposerContent))
					throw new ImportToolContentException("Import Noteice board tool content failed. Deserialized object is " + toolPOJO);
				BehaviourComposerContent toolContentObj = (BehaviourComposerContent) toolPOJO;
				
//				reset it to new toolContentId
				toolContentObj.setBehaviourComposerContentId(toolContentId);
				behaviourComposerContentDAO.saveBehaviourComposerContent(toolContentObj);
			} catch (ImportToolContentException e) {
				throw new ToolException(e);
			}
	}
	/* ===============Methods implemented from ToolSessionManager =============== */
	
	/** @see org.lamsfoundation.lams.tool.ToolSessionManager#createToolSession(java.lang.Long, java.lang.String, java.lang.Long) */
	public void createToolSession(Long toolSessionId, String toolSessionName, Long toolContentId) throws ToolException
	{
	    if (toolSessionId == null || toolContentId == null)
	    {
	        String error = "Failed to create tool session. The tool session id or tool content id is invalid";
	        throw new ToolException(error);
	    }
	    

	    if ((behaviourComposerContent = retrieveBehaviourComposer(toolContentId)) == null)
	    {
	        //use default content
		    BehaviourComposerContent defaultContent = retrieveBehaviourComposer(getToolDefaultContentIdBySignature(BehaviourComposerConstants.TOOL_SIGNATURE));
		   
		    if (defaultContent != null)
		    {
		        BehaviourComposerSession newSession = new BehaviourComposerSession(toolSessionId, 
		        														toolSessionName,
		                												defaultContent,
		                												new Date(System.currentTimeMillis()),
		                												BehaviourComposerSession.NOT_ATTEMPTED);
		        //saveBehaviourComposerSession(newSession);
		        defaultContent.getBehaviourComposerSessions().add(newSession);
		        saveBehaviourComposer(defaultContent);
		       
		    }
		    else
		    {
		        throw new ToolException("Default content is missing. Unable to create tool session");
		    }
	    }
	    else
	    {
	        BehaviourComposerSession behaviourComposerSession = new BehaviourComposerSession(toolSessionId,
	        									  toolSessionName,
												  behaviourComposerContent,
												  new Date(System.currentTimeMillis()),
												  BehaviourComposerSession.NOT_ATTEMPTED);
	        
	        behaviourComposerContent.getBehaviourComposerSessions().add(behaviourComposerSession);
	        saveBehaviourComposer(behaviourComposerContent);
	        //saveBehaviourComposerSession(behaviourComposerSession);
	    }
	    
	    
	    
	}
	
	/** @see org.lamsfoundation.lams.tool.ToolSessionManager#leaveToolSession(java.lang.Long, org.lamsfoundation.lams.usermanagement.User)*/
	public String leaveToolSession(Long toolSessionId, Long learnerId) throws DataMissingException, ToolException
	{
	    getAndCheckSessionIDandObject(toolSessionId);
	    
	   return learnerService.completeToolSession(toolSessionId, learnerId);
	}
   
	/** @see org.lamsfoundation.lams.tool.ToolSessionManager#exportToolSession(java.lang.Long)*/ 
	 public ToolSessionExportOutputData exportToolSession(Long toolSessionId) throws ToolException, DataMissingException
	{
	     getAndCheckSessionIDandObject(toolSessionId);
	    throw new UnsupportedOperationException("not yet implemented");
	}

	/** @see org.lamsfoundation.lams.tool.ToolSessionManager#exportToolSession(java.util.List) */
    public ToolSessionExportOutputData exportToolSession(List toolSessionIds) throws ToolException, DataMissingException
	{
        Iterator i = toolSessionIds.iterator();
        if (i.hasNext())
        {
            Long id = (Long)i.next();
            getAndCheckSessionIDandObject(id);
        }
        
        
	    throw new UnsupportedOperationException("not yet implemented");
	}
    
    /** @see org.lamsfoundation.lams.tool.ToolSessionManager#removeToolSession(java.lang.Long)*/
    public void removeToolSession(Long toolSessionId) throws DataMissingException, ToolException
    {
        BehaviourComposerSession session = getAndCheckSessionIDandObject(toolSessionId);
        removeSession(session);
    }
	
	/* ===============Methods implemented from ToolContentImport102Manager =============== */
	

    /**
     * Import the data for a 1.0.2 BehaviourComposer or HTMLBehaviourComposer
     */
    public void import102ToolContent(Long toolContentId, UserDTO user, Hashtable importValues)
    {
    	Date now = new Date();
    	BehaviourComposerContent toolContentObj = new BehaviourComposerContent();
    	toolContentObj.setContent((String)importValues.get(ToolContentImport102Manager.CONTENT_BODY));
    	toolContentObj.setContentInUse(false);
    	toolContentObj.setCreatorUserId(user.getUserID().longValue());
    	toolContentObj.setDateCreated(now);
    	toolContentObj.setDateUpdated(now);
    	toolContentObj.setDefineLater(false);
    	toolContentObj.setForceOffline(false);
    	toolContentObj.setBehaviourComposerContentId(toolContentId);
    	toolContentObj.setOfflineInstructions(null);
    	toolContentObj.setOnlineInstructions(null);
    	toolContentObj.setTitle((String)importValues.get(ToolContentImport102Manager.CONTENT_TITLE));
    	// leave as empty, no need to set them to anything.
    	//toolContentObj.setBehaviourComposerSessions(behaviourComposerSessions);
    	//toolContentObj.setBehaviourComposerAttachments(behaviourComposerAttachments);
    	behaviourComposerContentDAO.saveBehaviourComposerContent(toolContentObj);
    }

    /** Set the reflective title - does nothing as BehaviourComposer is not reflective */
    public void  setReflectiveData(Long toolContentId, String title, String defaultInputValues) 
    		throws ToolException, DataMissingException {
    	log.error("Was asked to set reflective data "+title+" to activity toolContentId "+toolContentId
    			+". BehaviourComposer does not support reflection");
    }
    
    //=========================================================================================
    
    public Long getToolDefaultContentIdBySignature(String toolSignature)
    {
        Long contentId = null;
    	contentId=new Long(toolService.getToolDefaultContentIdBySignature(toolSignature));    
    	if (contentId == null)
    	{
    	    String error="Could not retrieve default content id for this tool";
    	    log.error(error);
    	    throw new BehaviourComposerApplicationException(error);
    	}
	    return contentId;
    }

	/* getter setter methods to obtain the service bean */
	/*public IBehaviourComposerContentDAO getBehaviourComposerContentDAO()
	{
		return behaviourComposerContentDAO;
	}
	*/
	public void setBehaviourComposerContentDAO(IBehaviourComposerContentDAO behaviourComposerContentDAO)
	{
		this.behaviourComposerContentDAO = behaviourComposerContentDAO;
	}
	
	/*public IBehaviourComposerSessionDAO getBehaviourComposerSessionDAO()
	{
	    return behaviourComposerSessionDAO;
	}
	*/
	public void setBehaviourComposerSessionDAO(IBehaviourComposerSessionDAO behaviourComposerSessionDAO)
	{
	    this.behaviourComposerSessionDAO = behaviourComposerSessionDAO;
	}
	
	/*public IBehaviourComposerUserDAO getBehaviourComposerUserDAO()
	{
	    return behaviourComposerUserDAO;
	}
	*/
	public void setBehaviourComposerUserDAO(IBehaviourComposerUserDAO behaviourComposerUserDAO)
	{
	    this.behaviourComposerUserDAO = behaviourComposerUserDAO;
	}
	
    /**
     * @return Returns the learnerService.
     */
  /*  public ILearnerService getLearnerService() {
        return learnerService;
    } */
    /**
     * @param learnerService The learnerService to set.
     */
	 public void setLearnerService(ILearnerService learnerService) {
	        this.learnerService = learnerService;
	    } 
 /*   public IBehaviourComposerAttachmentDAO getBehaviourComposerAttachmentDAO() {
        return behaviourComposerAttachmentDAO;
    } */
   
    public void setBehaviourComposerAttachmentDAO(IBehaviourComposerAttachmentDAO behaviourComposerAttachmentDAO) {
        this.behaviourComposerAttachmentDAO = behaviourComposerAttachmentDAO;
    }
    /**
     * @param toolService The toolService to set.
     */
    public void setToolService(ILamsToolService toolService) {
        this.toolService = toolService;
    }

	public IToolContentHandler getBehaviourComposerToolContentHandler() {
		return behaviourComposerToolContentHandler;
	}

	public void setBehaviourComposerToolContentHandler(IToolContentHandler behaviourComposerToolContentHandler) {
		this.behaviourComposerToolContentHandler = behaviourComposerToolContentHandler;
	}
	public IExportToolContentService getExportContentService() {
		return exportContentService;
	}


	public void setExportContentService(IExportToolContentService exportContentService) {
		this.exportContentService = exportContentService;
	}
}
