/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: IBehaviourComposerSessionDAO.java,v 1.8 2006/04/06 07:05:07 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.behaviourComposer.dao;

import java.util.List;

import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerUser;


/**
 * <p>Interface for the BehaviourComposerSession DAO, defines methods needed to access/modify
 * behaviourComposer session</p>
 * @author mtruong
 */
public interface IBehaviourComposerSessionDAO {
	
   
    /**
	 * <p> Return the persistent instance of a BehaviourComposerSession
	 * with the given tool session id <code>behaviourComposerSessionId</code>,
	 * returns null if not found.</p>
	 * 
	 * @param behaviourComposerSessionId The tool session id
	 * @return the persistent instance of a BehaviourComposerSession or null if not found.
	 */
    public BehaviourComposerSession findBehaviourComposerSessionById(Long behaviourComposerSessionId);
	
	
    /**
	 * <p>Persist the given persistent instance of BehaviourComposerSession.</p>
	 * 
	 * @param behaviourComposerSession The instance of BehaviourComposerSession to persist.
	 */
    public void saveBehaviourComposerSession(BehaviourComposerSession behaviourComposerSession);
    
    /**
     * <p>Update the given persistent instance of BehaviourComposerSession.</p>
     * 
     * @param behaviourComposerContent The instance of BehaviourComposerSession to persist.
     */
    public void updateBehaviourComposerSession(BehaviourComposerSession behaviourComposerSession);
    
    
    /**
     * <p>Delete the given instance of BehaviourComposerSession</p>
     * 
     * @param behaviourComposerSession The instance of BehaviourComposerSession to delete. 
     */
    public void removeBehaviourComposerSession(BehaviourComposerSession behaviourComposerSession);
    
    /**
     * <p>Delete the given instance of BehaviourComposerSession with the
     * given tool session id <code>behaviourComposerSessionid</code>
     * 
     * @param behaviourComposerSessionId The tool session Id. 
     */
    public void removeBehaviourComposerSession(Long behaviourComposerSessionId);
    
    /**
     * <p> Returns the persistent instance of BehaviourComposerSession
     * associated with the given behaviourComposer user, with user id <code>userId</code>, 
     * returns null if not found.
     * 
     * @param userId The behaviourComposer user id
     * @return a persistent instance of BehaviourComposerSessions or null if not found.
     */	
    public BehaviourComposerSession getBehaviourComposerSessionByUser(Long userId); 
    
    /**
     * <p>Deletes all instances of BehaviourComposerUser that are associated
     * with the given instance of BehaviourComposerSession</p>
     * 
     * @param behaviourComposerSession The instance of BehaviourComposerSession in which corresponding instances of BehaviourComposerUser should be deleted.
     */
    public void removeBehaviourComposerUsers(BehaviourComposerSession behaviourComposerSession); 

    /**
     * <p>Creates and persists an instance of BehaviourComposerUser which is associated
     * with the BehaviourComposerSession with tool session id <code>behaviourComposerSessionId</code> </p>
     * 
     * @param behaviourComposerSessionId The tool session id
     * @param user The instance of BehaviourComposerUser
     */
    public void addBehaviourComposerUsers(Long behaviourComposerSessionId, BehaviourComposerUser user);
    
    /**
     * Returns a list of tool session ids which are associated with this
     * instance of BehaviourComposerContent. 
     * <p>For example, if the given instance <code>behaviourComposerContent</code> has a tool content id
     * of 3, and consider an extract of the tl_kkbc10_session table:</p>
     * <pre> 
     * 		 ----------------------------
     * 		 attachmentId | toolSessionId
     * 		 ----------------------------
     * 			1		  | 	3
     * 			2		  | 	3
     * 			3		  | 	3
     * 			4 		  | 	1
     * 		 ----------------------------
     * </pre>
     * Then a call to <code>getSessionsFromContent</code> will return a list containing the values
     * 1, 2 and 3. 
     * @param behaviourComposerContent The instance of BehaviourComposerContent in which you want the list of toolSessionIds
     * @return a list of tool session Ids
     */
    public List getSessionsFromContent(BehaviourComposerContent behaviourComposerContent);
}
