/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: BehaviourComposerSession.java,v 1.9 2006/04/06 07:05:07 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.behaviourComposer;

import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent;
import java.io.Serializable;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

/**
 * <p>The BehaviourComposerSession class represents a tool session for a behaviourComposer activity.
 * Each tool session may represent a group of users, if grouping is used, or may 
 * represent one learner if there is no grouping for this particular activity.
 * </p>
 * <br>
 * <p>The session status has three possible status':
 * <ul><li>NOT_ATTEMPTED: which means that the tool session has been established, but no learners have reached this activity yet 
 * <li>INCOMPLETE: which means that a learner has reached this activity 
 * <li>COMPLETED: The session status will never be set to complete as you don't know when this tool is going to end.
 * </ul></p>
 * @author mtruong
 * @hibernate.class table="tl_kkbc10_session"
 */

public class BehaviourComposerSession implements Serializable {

    
    public final static String NOT_ATTEMPTED = "NOT_ATTEMPTED";
    
    public final static String INCOMPLETE = "INCOMPLETE";
    
    public static final String COMPLETED = "COMPLETED";
    
	/** identifier field */
    private Long uid;
    
    /** persistent field*/
	private Long behaviourComposerSessionId;
	
	/** persistent field*/
	private String behaviourComposerSessionName;
	
  	/** nullable persistent field */
	private Date sessionStartDate;
		
	/** nullable persistent field */
	private Date sessionEndDate;
	
	/** nullable persistent field */
	private String sessionStatus;
	
	/** persistent field */
	private BehaviourComposerContent behaviourComposerContent;
	
	/** persistent field */
	private Set behaviourComposerUsers = new HashSet();
	
	/** default constructor */
	public BehaviourComposerSession()
	{
				
	}
	
	/** full constructor */
	public BehaviourComposerSession(Long behaviourComposerSessionId, 
							  String behaviourComposerSessionName,
							  BehaviourComposerContent behaviourComposerContent,
							  Date sessionStartDate,
							  Date sessionEndDate,
							  String sessionStatus)
	{
		this.behaviourComposerSessionId = behaviourComposerSessionId;
		this.behaviourComposerSessionName = behaviourComposerSessionName;
		//this.behaviourComposerContentId = behaviourComposerContentId;
		this.behaviourComposerContent = behaviourComposerContent;
		this.sessionStartDate = sessionStartDate;
		this.sessionEndDate = sessionEndDate;
		this.sessionStatus = sessionStatus;
	}
	
	/**
	 * Constructor used when creating a new  behaviourComposerSession given
	 * the behaviourComposerContent id
	 */
	
	public BehaviourComposerSession(Long behaviourComposerSessionId,
							  String behaviourComposerSessionName,
	        				  BehaviourComposerContent behaviourComposerContent,
	        				  Date sessionStartDate,
	        				  String sessionStatus)
	{
	    this.behaviourComposerSessionId = behaviourComposerSessionId;
	    this.behaviourComposerSessionName = behaviourComposerSessionName;
	    this.behaviourComposerContent = behaviourComposerContent;
	    this.sessionStartDate = sessionStartDate;
	    this.sessionEndDate = null;
	    this.sessionStatus = sessionStatus;
	}
	
	public BehaviourComposerSession(Long behaviourComposerSessionId, 
							  String behaviourComposerSessionName,
							  BehaviourComposerContent behaviourComposerContent)
	{
	    this.behaviourComposerSessionId = behaviourComposerSessionId;
	    this.behaviourComposerSessionName = behaviourComposerSessionName;
	    this.behaviourComposerContent = behaviourComposerContent;
	    this.sessionStartDate = new Date(System.currentTimeMillis());
		this.sessionStatus = BehaviourComposerSession.INCOMPLETE;
	}
	
	public BehaviourComposerSession(Long behaviourComposerSessionId, String behaviourComposerSessionName)
	{
	    this.behaviourComposerSessionId = behaviourComposerSessionId;
	    this.behaviourComposerSessionName = behaviourComposerSessionName;
	    this.sessionStartDate = new Date(System.currentTimeMillis());
		this.sessionStatus = BehaviourComposerSession.INCOMPLETE;
	}
	
	public BehaviourComposerSession(Long behaviourComposerSessionId)
	{
	    this.behaviourComposerSessionId = behaviourComposerSessionId;
	    this.sessionStartDate = new Date(System.currentTimeMillis());
		this.sessionStatus = BehaviourComposerSession.INCOMPLETE;
	}
	
	/**
	 * 		@hibernate.many-to-one
     *      not-null="true"
     *      @hibernate.column name="behaviourComposer_content_uid" 
	 */
	public BehaviourComposerContent getBehaviourComposerContent() {
		return behaviourComposerContent;
	}
	
	public void setBehaviourComposerContent(BehaviourComposerContent behaviourComposerContent) {
		this.behaviourComposerContent = behaviourComposerContent;
	}
		
	/**
	 *		@hibernate.property
     *      column="behaviourComposer_session_id"
     *      length="20"
     *      not-null="true"
	 */
	public Long getBehaviourComposerSessionId() {
		return behaviourComposerSessionId;
	}
	
	public void setBehaviourComposerSessionId(Long behaviourComposerSessionId) {
		this.behaviourComposerSessionId = behaviourComposerSessionId;
	}
	
	/**
	 * 		@hibernate.property
     *      column="behaviourComposer_session_name"
     *      length="255"
     *      not-null="true"
	 */
	public String getBehaviourComposerSessionName() {
		return behaviourComposerSessionName;
	}
	
	public void setBehaviourComposerSessionName(String behaviourComposerSessionName) {
		this.behaviourComposerSessionName = behaviourComposerSessionName;
	}
	
	/**
	 *		@hibernate.property
     *      column="session_end_date"
     *      length="19"
	 */
	public Date getSessionEndDate() {
		return sessionEndDate;
	}
	
	public void setSessionEndDate(Date sessionEndDate) {
		this.sessionEndDate = sessionEndDate;
	}
	
	/**
	 *		@hibernate.property
     *      column="session_start_date"
     *      length="19"
	 */
	public Date getSessionStartDate() {
		return sessionStartDate;
	}
	
	public void setSessionStartDate(Date sessionStartDate) {
		this.sessionStartDate = sessionStartDate;
	}
	
	/**
	 * 		@hibernate.property
     *      column="session_status"
     *      length="100"
	 */
	public String getSessionStatus() {
		return sessionStatus;
	}
	
	public void setSessionStatus(String sessionStatus) {
		this.sessionStatus = sessionStatus;
	}
	
	/**
     *	  	@hibernate.id
     *      generator-class="native"
     *      type="java.lang.Long"
     *      column="uid"
     *      unsaved-value="0"
     */
    public Long getUid() {
        return uid;
    }
    
    public void setUid(Long uid) {
        this.uid = uid;
    }
	
    /**
     * @hibernate.set
     *      lazy="true"
     *      inverse="true"
     *      cascade="all-delete-orphan"
  	 *
 	 * @hibernate.collection-key
	 * 		column="behaviourComposer_session_uid"
	 * @hibernate.collection-one-to-many
	 *      class="org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerUser"
     */
    public Set getBehaviourComposerUsers() {
        if (this.behaviourComposerUsers == null)
		{
			setBehaviourComposerUsers(new HashSet());
		}
    
        return behaviourComposerUsers;
    }
    /**
     * @param behaviourComposerUsers The behaviourComposerUsers to set.
     */
    public void setBehaviourComposerUsers(Set behaviourComposerUsers) {
        this.behaviourComposerUsers = behaviourComposerUsers;
    }
    
   
}
