/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: ExportServlet.java,v 1.6 2006/08/22 06:48:38 steven Exp $$ */	

package org.lamsfoundation.lams.tool.behaviourComposer.web;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.lamsfoundation.lams.contentrepository.FileException;
import org.lamsfoundation.lams.contentrepository.IVersionedNode;
import org.lamsfoundation.lams.contentrepository.NodeType;
import org.lamsfoundation.lams.tool.ToolAccessMode;
import org.lamsfoundation.lams.tool.behaviourComposer.Learner;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerContent;
import org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession;
import org.lamsfoundation.lams.tool.behaviourComposer.dto.FileDetailsDTO;
import org.lamsfoundation.lams.tool.behaviourComposer.exception.BehaviourComposerException;
import org.lamsfoundation.lams.tool.behaviourComposer.service.IBehaviourComposerService;
import org.lamsfoundation.lams.tool.behaviourComposer.service.BehaviourComposerServiceProxy;
import org.lamsfoundation.lams.usermanagement.dto.UserDTO;
import org.lamsfoundation.lams.usermanagement.util.LastNameAlphabeticComparator;
import org.lamsfoundation.lams.web.servlet.AbstractExportPortfolioServlet;

public class ExportServlet extends AbstractExportPortfolioServlet {
	private static final long serialVersionUID = -4529093489007108143L;

	private static Logger logger = Logger.getLogger(ExportServlet.class);

	private final String FILENAME = "behaviourComposer_main.html";
	
	private class StringComparator implements Comparator<String>{
		public int compare(String o1, String o2) {
			if(o1 != null && o2 != null){
				return o1.compareTo(o2);
			}else if(o1 != null)
				return 1;
			else
				return -1;
		}
	}
	public String doExport(HttpServletRequest request,
			HttpServletResponse response, String directoryName, Cookie[] cookies) {
		
		IBehaviourComposerService behaviourComposerService = BehaviourComposerServiceProxy
				.getBehaviourComposerService(getServletContext());

		Map map = null;
		if (StringUtils.equals(mode, ToolAccessMode.LEARNER.toString())) {
			map = learner(request, response, directoryName, cookies, behaviourComposerService);
		} else if (StringUtils.equals(mode, ToolAccessMode.TEACHER.toString())) {
			map = teacher(request, response, directoryName, cookies, behaviourComposerService);
		}

		String basePath = request.getScheme() + "://" + request.getServerName()
				+ ":" + request.getServerPort() + request.getContextPath();

		// Writing learner behaviourComposerted files to disk. A folder is generated for
		// each
		// learner, within this folder a sub folder is generated for each
		// behaviourComposerted file. The FileDetailsDTO object is updated to record the
		// filepath of the exported file.
		
		List attachmentList = new LinkedList();
		
		Iterator userIter = map.keySet().iterator();
		while (userIter.hasNext()) {
			UserDTO user = (UserDTO) userIter.next();
			List fileList = (List) map.get(user);

			Iterator fileListIter = fileList.iterator();

			int submissionCount = 1;

			while (fileListIter.hasNext()) {

				FileDetailsDTO behaviourComposertedFile = (FileDetailsDTO) fileListIter
						.next();

				logger.debug("doExport: starting to write file:"
						+ behaviourComposertedFile.getFilePath());

				IVersionedNode node = behaviourComposerService.downloadFile(behaviourComposertedFile
						.getUuID(), behaviourComposertedFile.getVersionID());
				
				// checking to see if our node type is a file.
				if (node.isNodeType(NodeType.FILENODE)) {
					try {
						InputStream is = node.getFile();
						BufferedInputStream bis = new BufferedInputStream(is);
						
						// The path name where this file is to be stored
						File outFileDir = new File(directoryName
								+ File.separator + user.getLogin()
								+ File.separator + submissionCount);

						// Creating the directory structure
						outFileDir.mkdirs();

						File outFile = new File(outFileDir, behaviourComposertedFile
								.getFilePath());

						// writing the data to file
						FileOutputStream fos = new FileOutputStream(outFile);
						BufferedOutputStream bos = new BufferedOutputStream(fos);
						
						for (int c = bis.read(); c != -1; c = bis.read()) {
							bos.write(c);
						}

						logger.debug("doExport: finished writing to file");
						
						is.close();
						fos.close();

					} catch (FileException fe) {
						String error = "Unable to retrieve file from repository";
						logger.error(error);
						throw new BehaviourComposerException(error);
					} catch (IOException ioe) {
						String error = "An IO error occured while writing file to disk";
						logger.error(error);
						throw new BehaviourComposerException(error);
					}
				} else {
					String error = "Node in repository is not a FILENODE";
					logger.error(error);
					throw new BehaviourComposerException(error);
				}
				
				logger.debug("doExport: encoding exportedURL");
				
				String exportedURL = user.getLogin() + "/" + submissionCount
						+ "/" + behaviourComposertedFile.getFilePath();
				
				attachmentList.add(exportedURL);
				
				// encoding the URL to application/x-www-form-urlencoded
				// see
				// http://www.w3.org/TR/html4/interact/forms.html#h-17.13.4.1
				try {
					URLEncoder.encode(exportedURL, "UTF8");
				} catch (UnsupportedEncodingException e) {
					exportedURL = "/";
					String error = "Unable to URL encode the file path, files written to disk";
					logger.error(error);
					throw new BehaviourComposerException(error);

				}

				exportedURL = exportedURL.replaceAll("\\+", "%20");
				behaviourComposertedFile.setExportedURL(exportedURL);
				submissionCount++;

				logger.debug("doExport: finished file:"
						+ behaviourComposertedFile.getFilePath());
			}
		}

		// Generating the attachments.txt file. This contains a list of all
		// files exported.
		// TODO
		
		// writing out the attachmentList
		request.getSession().setAttribute("attachmentList", attachmentList);
		writeResponseToFile(basePath + "/export/exportAttachmentList.jsp",
				directoryName, "attachment_list.txt", cookies);
		
		// generate the behaviourComposer main page
		writeResponseToFile(basePath + "/export/exportportfolio.jsp",
				directoryName, FILENAME, cookies);

		return FILENAME;

	}

	public Map learner(HttpServletRequest request,
			HttpServletResponse response, String directoryName, Cookie[] cookies, IBehaviourComposerService behaviourComposerService) {
		
		if (userID == null || toolSessionID == null) {
			String error = "Tool session Id or user Id is null. Unable to continue";
			logger.error(error);
			throw new BehaviourComposerException(error);
		}

		Learner learner = behaviourComposerService.getLearner(toolSessionID, userID);

		if (learner == null) {
			String error = "The user with user id "
					+ userID
					+ " does not exist in this session or session may not exist.";
			logger.error(error);
			throw new BehaviourComposerException(error);
		}

		BehaviourComposerContent content = behaviourComposerService
				.getBehaviourComposerContent(toolContentID);

		if (content == null) {
			String error = "The content for this activity has not been defined yet.";
			logger.error(error);
			throw new BehaviourComposerException(error);
		}

		List fileList = behaviourComposerService.getFilesUploadedByUser(userID,toolSessionID);
		//if mark not release, then set these message as null.
		Iterator iter = fileList.iterator();
		while(iter.hasNext()){
			FileDetailsDTO filedto = (FileDetailsDTO) iter.next();
			if(filedto .getDateMarksReleased() == null){
				filedto .setComments(null);
				filedto .setMarks(null);
			}
		}
		
		Map userFilesMap = new HashMap();
		userFilesMap.put(behaviourComposerService.getUserDetails(learner.getUserID()),fileList);
		
		//add session name to construct a new map
		Map report = new TreeMap(this.new StringComparator());
		report.put(behaviourComposerService.getSessionById(toolSessionID).getSessionName(), userFilesMap);
		request.getSession().setAttribute("report", report);
		return userFilesMap;
	}

	public Map teacher(HttpServletRequest request,
			HttpServletResponse response, String directoryName, Cookie[] cookies, IBehaviourComposerService behaviourComposerService) {
		
		// check if toolContentId exists in db or not
		if (toolContentID == null) {
			String error = "Tool Content Id is missing. Unable to continue";
			logger.error(error);
			throw new BehaviourComposerException(error);
		}

		BehaviourComposerContent content = behaviourComposerService
				.getBehaviourComposerContent(toolContentID);

		if (content == null) {
			String error = "Data is missing from the database. Unable to Continue";
			logger.error(error);
			throw new BehaviourComposerException(error);
		}
		// return FileDetailsDTO list according to the given sessionID
		Set sessionList = content.getToolSession();
		Iterator iter = sessionList.iterator();
		Map report = new TreeMap(this.new StringComparator());
		Map allFileMap = new TreeMap(new LastNameAlphabeticComparator());
		while (iter.hasNext()) {
			SortedMap userFilesMap = new TreeMap(new LastNameAlphabeticComparator());
			BehaviourComposerSession session = (BehaviourComposerSession) iter.next();
			userFilesMap.putAll(behaviourComposerService.getFilesUploadedBySession(session
					.getSessionID()));
			allFileMap.putAll(userFilesMap);
			report.put(session.getSessionName(), userFilesMap);
		}
		
//		add session name to construct a new map
		request.getSession().setAttribute("report", report);

		return allFileMap;
	}

}
