/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: BehaviourComposerContent.java,v 1.14 2006/08/17 02:16:07 steven Exp $$ */	
package org.lamsfoundation.lams.tool.behaviourComposer;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.log4j.Logger;
import org.lamsfoundation.lams.contentrepository.ItemNotFoundException;
import org.lamsfoundation.lams.contentrepository.NodeKey;
import org.lamsfoundation.lams.contentrepository.RepositoryCheckedException;
import org.lamsfoundation.lams.contentrepository.client.IToolContentHandler;


/**
 * @hibernate.class table="tl_kkbc10_content"
 * @serial 9072799761861936838L
 */
public class BehaviourComposerContent implements Serializable,Cloneable {

	private static final long serialVersionUID = 9072799761861936838L;

	private static Logger log = Logger.getLogger(BehaviourComposerContent.class);
	/** identifier field */
	private Long contentID;

	/** persistent field */
	//basic tab fields
	private String title;
	private String instruction;
	
	// advance tab fields
	private boolean lockOnFinished;
	
	//instruction tab fields
	private String offlineInstruction;
	private String onlineInstruction;
	private Set instructionFiles;
	
	//system level fields
	private boolean defineLater;
	private boolean runOffline;
	private boolean contentInUse;

	//relationship fields
	private Set toolSession;
	//temporary fields
	private IToolContentHandler toolContentHandler;
	
	/** full constructor */
	public BehaviourComposerContent(String title, String instructions,
							  Set toolSession) {
		this.title = title;
		this.instruction = instructions;		
		this.toolSession = toolSession;
	}

	/** default constructor */
	public BehaviourComposerContent() {
	}

	/** minimal constructor */
	public BehaviourComposerContent(Long contentID, 
							  String title,
							  String instructions,
							  Set toolSession
							  ) {
		super();
		this.contentID = contentID;
		this.title = title;
		this.instruction = instructions;
		this.toolSession = toolSession;
	}
	
	public BehaviourComposerContent(Long contentID, String title, String instructions){
		this.contentID = contentID;
		this.title = title;
		this.instruction = instructions;
	}

	/**
	 * Copy constructor to create a new BehaviourComposer tool's content.
	 * 
	 * @param content The original tool content
	 * @param newContentID The new <code>BehaviourComposer</code> contentID
	 * @param toolContentHandler 
	 * @return BehaviourComposerContent The new BehaviourComposerContent object
	 */
	public static BehaviourComposerContent newInstance(BehaviourComposerContent content,
			Long newContentID, IToolContentHandler toolContentHandler) {
		content.toolContentHandler  = toolContentHandler;
		BehaviourComposerContent newContent = (BehaviourComposerContent) content.clone();
		newContent.setContentID(newContentID);
		return newContent;
	}

	/**
	 * @hibernate.id generator-class="assigned" 
	 * 				 type="java.lang.Long"
	 *               column="content_id"
	 */
	public Long getContentID() {
		return this.contentID;
	}

	public void setContentID(Long contentID) {
		this.contentID = contentID;
	}

	/**
	 * @hibernate.property column="title" length="64" not-null="true"
	 */
	public String getTitle() {
		return this.title;
	}

	public void setTitle(String title) {
		this.title = title;
	}

	/**
	 * @hibernate.property column="instruction" type="text"
	 */
	public String getInstruction() {
		return this.instruction;
	}

	public void setInstruction(String instructions) {
		this.instruction = instructions;
	}	

	/**
	 * @hibernate.set lazy="true" inverse="true" cascade="all-delete-orphan"
	 * @hibernate.collection-key column="content_id"
	 * @hibernate.collection-one-to-many class="org.lamsfoundation.lams.tool.behaviourComposer.BehaviourComposerSession"
	 *  
	 */
	public Set getToolSession() {
		return this.toolSession;
	}

	public void setToolSession(Set toolSession) {
		this.toolSession = toolSession;
	}

	public String toString() {
		return new ToStringBuilder(this).append("contentID", getContentID())
				.append("title", getTitle()).append("instructions",
						getInstruction()).toString();
	}

	public boolean equals(Object other) {
		if ((this == other))
			return true;
		if (!(other instanceof BehaviourComposerContent))
			return false;
		BehaviourComposerContent castOther = (BehaviourComposerContent) other;
		return new EqualsBuilder().append(this.getContentID(),
				castOther.getContentID()).append(this.getTitle(),
				castOther.getTitle()).append(this.getInstruction(),
				castOther.getInstruction()).isEquals();
	}

	public int hashCode() {
		return new HashCodeBuilder().append(getContentID()).append(getTitle())
				.append(getInstruction()).toHashCode();
	}


    /** 
     * @hibernate.property column="define_later" length="1"
     *  not-null="true"
     */
    public boolean isDefineLater()
    {
        return this.defineLater;
    }

    public void setDefineLater(boolean defineLater)
    {
        this.defineLater = defineLater;
    }

    /** 
     * @hibernate.property column="run_offline" length="1"
     * not-null="true"
     */
    public boolean isRunOffline()
    {
        return this.runOffline;
    }

    public void setRunOffline(boolean runOffline)
    {
        this.runOffline = runOffline;
    }

	/**
	 * @hibernate.property column="offline_instruction" type="text"
	 * @return Returns the offlineInstruction.
	 */
	public String getOfflineInstruction() {
		return offlineInstruction;
	}

	/**
	 * @param offlineInstruction The offlineInstruction to set.
	 */
	public void setOfflineInstruction(String offlineInstruction) {
		this.offlineInstruction = offlineInstruction;
	}

	/**
	 * @hibernate.property column="online_instruction" type="text"
	 * @return Returns the onlineInstruction.
	 */
	public String getOnlineInstruction() {
		return onlineInstruction;
	}

	/**
	 * @param onlineInstruction The onlineInstruction to set.
	 */
	public void setOnlineInstruction(String onlineInstruction) {
		this.onlineInstruction = onlineInstruction;
	}

	/**
	 * @hibernate.property column="content_in_use" length="1"
	 * @return Returns the contentInUse.
	 */
	public boolean isContentInUse() {
		return contentInUse;
	}

	/**
	 * @param contentInUse The contentInUse to set.
	 */
	public void setContentInUse(boolean contentInUse) {
		this.contentInUse = contentInUse;
	}

	/**
 	 * @hibernate.set lazy="true" inverse="false" cascade="all-delete-orphan"
	 * @hibernate.collection-key column="content_id"
	 * @hibernate.collection-one-to-many class="org.lamsfoundation.lams.tool.behaviourComposer.InstructionFiles"
	 * 
	 * @return Returns the instructionFiles.
	 */
	public Set getInstructionFiles() {
		return instructionFiles;
	}

	/**
	 * @param instructionFiles The instructionFiles to set.
	 */
	public void setInstructionFiles(Set instructionFiles) {
		this.instructionFiles = instructionFiles;
	}

	/**
	 * 
	 * @hibernate.property column="lock_on_finished" length="1"
	 * @return Returns the lockOnFinished.
	 */
	public boolean isLockOnFinished() {
		return lockOnFinished;
	}

	/**
	 * @param lockOnFinished The lockOnFinished to set.
	 */
	public void setLockOnFinished(boolean lockOnFinished) {
		this.lockOnFinished = lockOnFinished;
	}

    
    public Object clone(){
		Object obj = null;
		try {
			obj = super.clone();
			//never clone key!
			((BehaviourComposerContent)obj).setContentID(null);
			//clone BehaviourComposerFileSession object
			if(toolSession != null ){
				Iterator iter = toolSession.iterator();
				Set set = new HashSet();
				while(iter.hasNext())
					set.add(((BehaviourComposerSession)iter.next()).clone());
				((BehaviourComposerContent)obj).toolSession = set;
			}
			//clone InstructionFiles object
			if(instructionFiles != null ){
				Iterator iter = instructionFiles.iterator();
				Set set = new HashSet();
				while(iter.hasNext()){
					InstructionFiles file = (InstructionFiles)iter.next();
					InstructionFiles newFile = (InstructionFiles) file.clone();
					//duplicate file node in repository
					if(toolContentHandler != null){
						NodeKey keys = toolContentHandler.copyFile(file.getUuID());
						newFile.setUuID(keys.getUuid());
						newFile.setVersionID(keys.getVersion());
					}
					set.add(newFile);
				}
				((BehaviourComposerContent)obj).instructionFiles= set;
			}
		} catch (CloneNotSupportedException e) {
			log.error("When clone " + BehaviourComposerContent.class + " failed");
		} catch (ItemNotFoundException e) {
			log.error("When clone " + BehaviourComposerContent.class + " failed");
		} catch (RepositoryCheckedException e) {
			log.error("When clone " + BehaviourComposerContent.class + " failed");
		}
		
		return obj;
	}

	public IToolContentHandler getToolContentHandler() {
		return toolContentHandler;
	}

	public void setToolContentHandler(IToolContentHandler toolContentHandler) {
		this.toolContentHandler = toolContentHandler;
	}

}
