/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: ITwitterSessionDAO.java,v 1.9 2006/09/17 06:26:26 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.twitter.dao;

import java.util.List;

import org.lamsfoundation.lams.tool.twitter.TwitterContent;
import org.lamsfoundation.lams.tool.twitter.TwitterSession;
import org.lamsfoundation.lams.tool.twitter.TwitterUser;


/**
 * <p>Interface for the TwitterSession DAO, defines methods needed to access/modify
 * twitter session</p>
 * @author mtruong
 */
public interface ITwitterSessionDAO {
	
   
    /**
	 * <p> Return the persistent instance of a TwitterSession
	 * with the given tool session id <code>twitterSessionId</code>,
	 * returns null if not found.</p>
	 * 
	 * @param twitterSessionId The tool session id
	 * @return the persistent instance of a TwitterSession or null if not found.
	 */
    public TwitterSession findTwitterSessionById(Long twitterSessionId);
	
	
    /**
	 * <p>Persist the given persistent instance of TwitterSession.</p>
	 * 
	 * @param twitterSession The instance of TwitterSession to persist.
	 */
    public void saveTwitterSession(TwitterSession twitterSession);
    
    /**
     * <p>Update the given persistent instance of TwitterSession.</p>
     * 
     * @param twitterContent The instance of TwitterSession to persist.
     */
    public void updateTwitterSession(TwitterSession twitterSession);
    
    
    /**
     * <p>Delete the given instance of TwitterSession</p>
     * 
     * @param twitterSession The instance of TwitterSession to delete. 
     */
    public void removeTwitterSession(TwitterSession twitterSession);
    
    /**
     * <p>Delete the given instance of TwitterSession with the
     * given tool session id <code>twitterSessionid</code>
     * 
     * @param twitterSessionId The tool session Id. 
     */
    public void removeTwitterSession(Long twitterSessionId);
    
    /**
     * <p> Returns the persistent instance of TwitterSession
     * associated with the given twitter user, with user id <code>userId</code>, 
     * returns null if not found.
     * 
     * @param userId The twitter user id
     * @return a persistent instance of TwitterSessions or null if not found.
     */	
    public TwitterSession getTwitterSessionByUser(Long userId); 
    
    /**
     * <p>Deletes all instances of TwitterUser that are associated
     * with the given instance of TwitterSession</p>
     * 
     * @param twitterSession The instance of TwitterSession in which corresponding instances of TwitterUser should be deleted.
     */
    public void removeTwitterUsers(TwitterSession twitterSession); 

    /**
     * <p>Creates and persists an instance of TwitterUser which is associated
     * with the TwitterSession with tool session id <code>twitterSessionId</code> </p>
     * 
     * @param twitterSessionId The tool session id
     * @param user The instance of TwitterUser
     */
    public void addTwitterUsers(Long twitterSessionId, TwitterUser user);
    
    /**
     * Returns a list of tool session ids which are associated with this
     * instance of TwitterContent. 
     * <p>For example, if the given instance <code>twitterContent</code> has a tool content id
     * of 3, and consider an extract of the tl_fptwit10_session table:</p>
     * <pre> 
     * 		 ----------------------------
     * 		 attachmentId | toolSessionId
     * 		 ----------------------------
     * 			1		  | 	3
     * 			2		  | 	3
     * 			3		  | 	3
     * 			4 		  | 	1
     * 		 ----------------------------
     * </pre>
     * Then a call to <code>getSessionsFromContent</code> will return a list containing the values
     * 1, 2 and 3. 
     * @param twitterContent The instance of TwitterContent in which you want the list of toolSessionIds
     * @return a list of tool session Ids
     */
    public List getSessionsFromContent(TwitterContent twitterContent);
}
