/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: TwitterSession.java,v 1.10 2006/09/17 06:26:26 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.twitter;

import org.lamsfoundation.lams.tool.twitter.TwitterContent;
import java.io.Serializable;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

/**
 * <p>The TwitterSession class represents a tool session for a twitter activity.
 * Each tool session may represent a group of users, if grouping is used, or may 
 * represent one learner if there is no grouping for this particular activity.
 * </p>
 * <br>
 * <p>The session status has three possible status':
 * <ul><li>NOT_ATTEMPTED: which means that the tool session has been established, but no learners have reached this activity yet 
 * <li>INCOMPLETE: which means that a learner has reached this activity 
 * <li>COMPLETED: The session status will never be set to complete as you don't know when this tool is going to end.
 * </ul></p>
 * @author mtruong
 * @hibernate.class table="tl_fptwit10_session"
 */

public class TwitterSession implements Serializable {

    
    public final static String NOT_ATTEMPTED = "NOT_ATTEMPTED";
    
    public final static String INCOMPLETE = "INCOMPLETE";
    
    public static final String COMPLETED = "COMPLETED";
    
	/** identifier field */
    private Long uid;
    
    /** persistent field*/
	private Long twitterSessionId;
	
	/** persistent field*/
	private String twitterSessionName;
	
  	/** nullable persistent field */
	private Date sessionStartDate;
		
	/** nullable persistent field */
	private Date sessionEndDate;
	
	/** nullable persistent field */
	private String sessionStatus;
	
	/** persistent field */
	private TwitterContent twitterContent;
	
	/** persistent field */
	private Set twitterUsers = new HashSet();
	
	/** default constructor */
	public TwitterSession()
	{
				
	}
	
	/** full constructor */
	public TwitterSession(Long twitterSessionId, 
							  String twitterSessionName,
							  TwitterContent twitterContent,
							  Date sessionStartDate,
							  Date sessionEndDate,
							  String sessionStatus)
	{
		this.twitterSessionId = twitterSessionId;
		this.twitterSessionName = twitterSessionName;
		//this.twitterContentId = twitterContentId;
		this.twitterContent = twitterContent;
		this.sessionStartDate = sessionStartDate;
		this.sessionEndDate = sessionEndDate;
		this.sessionStatus = sessionStatus;
	}
	
	/**
	 * Constructor used when creating a new  twitterSession given
	 * the twitterContent id
	 */
	
	public TwitterSession(Long twitterSessionId,
							  String twitterSessionName,
	        				  TwitterContent twitterContent,
	        				  Date sessionStartDate,
	        				  String sessionStatus)
	{
	    this.twitterSessionId = twitterSessionId;
	    this.twitterSessionName = twitterSessionName;
	    this.twitterContent = twitterContent;
	    this.sessionStartDate = sessionStartDate;
	    this.sessionEndDate = null;
	    this.sessionStatus = sessionStatus;
	}
	
	public TwitterSession(Long twitterSessionId, 
							  String twitterSessionName,
							  TwitterContent twitterContent)
	{
	    this.twitterSessionId = twitterSessionId;
	    this.twitterSessionName = twitterSessionName;
	    this.twitterContent = twitterContent;
	    this.sessionStartDate = new Date(System.currentTimeMillis());
		this.sessionStatus = TwitterSession.INCOMPLETE;
	}
	
	public TwitterSession(Long twitterSessionId, String twitterSessionName)
	{
	    this.twitterSessionId = twitterSessionId;
	    this.twitterSessionName = twitterSessionName;
	    this.sessionStartDate = new Date(System.currentTimeMillis());
		this.sessionStatus = TwitterSession.INCOMPLETE;
	}
	
	public TwitterSession(Long twitterSessionId)
	{
	    this.twitterSessionId = twitterSessionId;
	    this.sessionStartDate = new Date(System.currentTimeMillis());
		this.sessionStatus = TwitterSession.INCOMPLETE;
	}
	
	/**
	 * 		@hibernate.many-to-one
     *      not-null="true"
     *      @hibernate.column name="twitter_content_uid" 
	 */
	public TwitterContent getTwitterContent() {
		return twitterContent;
	}
	
	public void setTwitterContent(TwitterContent twitterContent) {
		this.twitterContent = twitterContent;
	}
		
	/**
	 *		@hibernate.property
     *      column="twitter_session_id"
     *      length="20"
     *      not-null="true"
	 */
	public Long getTwitterSessionId() {
		return twitterSessionId;
	}
	
	public void setTwitterSessionId(Long twitterSessionId) {
		this.twitterSessionId = twitterSessionId;
	}
	
	/**
	 * 		@hibernate.property
     *      column="twitter_session_name"
     *      length="255"
     *      not-null="true"
	 */
	public String getTwitterSessionName() {
		return twitterSessionName;
	}
	
	public void setTwitterSessionName(String twitterSessionName) {
		this.twitterSessionName = twitterSessionName;
	}
	
	/**
	 *		@hibernate.property
     *      column="session_end_date"
     *      length="19"
	 */
	public Date getSessionEndDate() {
		return sessionEndDate;
	}
	
	public void setSessionEndDate(Date sessionEndDate) {
		this.sessionEndDate = sessionEndDate;
	}
	
	/**
	 *		@hibernate.property
     *      column="session_start_date"
     *      length="19"
	 */
	public Date getSessionStartDate() {
		return sessionStartDate;
	}
	
	public void setSessionStartDate(Date sessionStartDate) {
		this.sessionStartDate = sessionStartDate;
	}
	
	/**
	 * 		@hibernate.property
     *      column="session_status"
     *      length="100"
	 */
	public String getSessionStatus() {
		return sessionStatus;
	}
	
	public void setSessionStatus(String sessionStatus) {
		this.sessionStatus = sessionStatus;
	}
	
	/**
     *	  	@hibernate.id
     *      generator-class="native"
     *      type="java.lang.Long"
     *      column="uid"
     *      unsaved-value="0"
     */
    public Long getUid() {
        return uid;
    }
    
    public void setUid(Long uid) {
        this.uid = uid;
    }
	
    /**
     * @hibernate.set
     *      lazy="true"
     *      inverse="true"
     *      cascade="all-delete-orphan"
  	 *
 	 * @hibernate.collection-key
	 * 		column="twitter_session_uid"
	 * @hibernate.collection-one-to-many
	 *      class="org.lamsfoundation.lams.tool.twitter.TwitterUser"
     */
    public Set getTwitterUsers() {
        if (this.twitterUsers == null)
		{
			setTwitterUsers(new HashSet());
		}
    
        return twitterUsers;
    }
    /**
     * @param twitterUsers The twitterUsers to set.
     */
    public void setTwitterUsers(Set twitterUsers) {
        this.twitterUsers = twitterUsers;
    }
    
   
}
